// Low Frequency Rotary Magnetic Field Exciter Software (Dual 2 plane Helmholtz coils)
// PCB RE-03 (new MCU MSP430FR5739 16KB FRAM)
// this version is for PEMF 265Hz application with frequency adjust capability(but not used)
//Rev 03 adds 48V support, dual DC/DC converters with Shift register voltage control

// P1.0 is clk output (which loops to P1.1 input for further timer division)
// P2.5(pre), P2.6(clr)
// P2.7(pre), P2.4(clr)
// I2C P1.6-SDA and P1.7-SCL
// HB enables, P1.5 and P1.2
// ADC inputs P1.4(A4) and P2.3(A6)
// LM317 control P3.0 output
// P3.1(data), P3.2(data), and P3.3(clock) output to shift registers
// P3.5, P3.6, and P3.7 not used
// P1.3, P3.4 phase inputs
// LCD reset P2.2 output
// Switch inputs P2.0, P2.1, P4.1
// PJ.4 --> XIN, PJ4.5 XOUT

// uses 24 MHz external XTAL oscillator for improved stability/accuracy

//DCOCLK   (23.8MHz +/-2% when 0-50degC)
//ACLK    24MHz Max  (>3V)
//SMCLK   24MHz Max  (>3V)
//MCLK    24MHz max (>3V)
//ADC10CLK typ 4.5 MHz (5.5MHz max)
//VLO 8.3KHz typ, 13KHz max

//  ** to enable hex output for command line programming
// Project -> Properties -> MSP430 Hex Utility
// (check the box for Enable MSP430 Hex Utility)
// Click on Output Format Options
// Select TI-TXT from the list.
/*
                Kohm    VOLTS
1   00000001    66.67   4.93
4   00000100    50.00   6.16
8   00001000    42.86   6.98
5   00000101    40.00   7.39
9   00001001    35.29   8.21
10  00010000    28.67   9.83
11  00010001    25.08   11.06
12  00010010    22.28   12.29
0f  00001111    20.69   13.14
13  00010011    20.05   13.52
19  00011001    18.79   14.34
16  00010110    18.22   14.75
20  00100000    17.70   15.16
21  00100001    16.26   16.39
23  00100011    13.98   18.85
29  00101001    13.36   19.67
30  00110000    12.12   21.56
31  00110001    11.43   22.79
2E  00101110    11.13   23.37
34  00110100    10.81   24.02
38  00111000    10.44   24.84
39  00111001    9.92    26.08
3A  00111010    9.45    27.31
3C  00111100    9.45    27.31
40  01000000    9.09    28.34
3D  00111101    9.02    28.54
3E  00111110    8.63    29.77
48  01001000    8.11    31.62
4A  01001010    7.50    34.09
50  01010000    7.41    34.47
4B  01001011    7.23    35.32
51  01010001    7.15    35.70
4F  01001111    6.74    37.78
58  01011000    6.75    37.75
60  01100000    6.39    39.80
5c  01011100    6.32    40.21
62  01100010    6.01    42.26
5F  01011111    5.77    43.91
6A  01101010    5.56    45.55
70  01110000    5.51    45.93
6B  01101011    5.41    46.78
6E  01101110    5.27    48.01
6F  01101111    5.13    49.24
79  01111001    5.01    50.44
7A  01111010    4.88    51.68
7B  01111011    4.77    52.91
7E  01111110    4.66    54.14
7F  01111111    4.55    55.37
C3  11000011    4.44    56.67
C6  11000110    4.35    57.90
C7  11000111    4.26    59.14
CB  11001011    4.20    59.96
*/

#include <msp430.h>
#include <stdint.h>
#include <stdbool.h>

//Defines the port to be used for I2C
#define I2C_MASTER_REN  P1REN
#define I2C_MASTER_DIR  P1DIR
#define I2C_MASTER_OUT  P1OUT
#define I2C_MASTER_IN   P1IN
#define SDApin          BIT6
#define SCLpin          BIT7
//other defines
#define HBen (BIT2|BIT5)                //two enable bits (on port 1)
#define FBenLoop BIT2                   //Loop gate drive en
#define FBenSpiral BIT5                //Spiral gate drive en
#define LM317 BIT0                      // on port 3
#define cM  24                          //24MHz clock rate of MSP430
#define bUP BIT0                        //up button pin
#define bDW BIT1                        //down button pin
#define sPIN P1IN&BIT3                  //Spiral Coil Phase PIN
#define lPIN P3IN&BIT4                  //Loop Coil Phase PIN

//-----------[ Functions' Prototypes ]--------------
void delay_ms(unsigned int xdms);
void i2cm_assignPins(unsigned char unit);
void i2cm_start(void);
void i2cm_stop(void);
unsigned char i2cm_out(register unsigned int data );
void numL2_ascii(int32_t vp);               //large number (8 digits max) to ASCII
void init_LCD(void);
void startUP_LCD(void);
void char2_HEX(int8_t vp);

unsigned char flg=1;
signed char xt;                     //reserved for timer use
unsigned int xs;                     //shift register use
unsigned int x;
unsigned int y;

unsigned int cntMain=1;             //count main loops
unsigned int cntMainPrev=0;
unsigned int b;                     //for next loop
uint16_t resultADC;                 //ADC result (2 bytes)
uint32_t Vt1a, Vt2a, Vt1b, Vt2b;    //sum of ADC reading for purposes of averaging (max 256 averages)
uint32_t V1a, V2a, V1b, V2b;        //voltage measurements
uint32_t V1pk;                      //#1 high voltage
uint32_t vprev;                     //prev voltage for tuning resonance
uint16_t phase, phase1, phase2, phase3, phasePrev, ph2a, ph2b, ph2c;          //phase time values
uint32_t phs1, phf1, phs2, phf2, phs3, phf3, phCyc1, phCyc2, phCyc3;

int32_t data[6];                    //I2C received data (requires 32bit to support bit shifting)
uint8_t t8;                         //temporary 8 bit holder
int16_t t16;                        //temporary 16 bit holder
int32_t t32;                        //temporary 32 bit holder
uint8_t asc3, asc2, asc1;
uint8_t pC[10];                     //characters for display
int16_t freq;                       //temporary frequency for display
int16_t dutyCycle;                  //temporary  for display
float tfloat;                       //temporary for float point calculations

//#define wdt_start      (WDTPW + WDTTMSEL + WDTCNTCL + WDTSSEL__VLO + WDTIS__32K)  //  *** WATCHDOG TIMER MODE!!!!!  100us*WDTIS__32K= 3.2secs (for test)
#define wdt_start    (WDTPW + WDTTMSEL + WDTCNTCL + WDTSSEL__VLO + WDTIS__512K)     //  *** WATCHDOG TIMER MODE!!!!!  100us*WDTIS__512K= 51.2secs

// &&&&&&&&&&&&& Parameters to configure &&&&&&&&&&&&&&
#define  fTarget    530                //must be twice the desired freq due to the external /2 d flip-flop
//#define  fTarget    2162                //must be twice the desired freq due to the external /2 d flip-flop
#define onTimeMS    40                  //in msec : *note time likely needs to be 10ms or larger or the code delay for ADC must be adjusted!
                                        //40ms min for freq of 265Hz (or not stable/phase jumps)
const int  onTime= (onTimeMS/1000.0 * fTarget);    // on time in seconds  (last # is the target freq of resonance)
const int  offTime1= (0.2 * fTarget);  // off time in seconds  (last # is the target freq of resonance)
const int  offTime2= (0.45 * fTarget);  // off time in seconds  (last # is the target freq of resonance)
const int  offTime3= (0.95 * fTarget);  // off time in seconds  (last # is the target freq of resonance)
const int  offTime4= (1.45 * fTarget);  // off time in seconds  (last # is the target freq of resonance)
const int  offTime5= (1.95 * fTarget);  // off time in seconds  (last # is the target freq of resonance)
const int  offTime6= (4 * fTarget);  // off time in seconds  (last # is the target freq of resonance)
const int  offTime7= (8 * fTarget);  // off time in seconds  (last # is the target freq of resonance)
const int  tuneManStep = 60;           //manual tune step size (push buttons)
const int  tuneAutoStep = 0;           //0 disables the auto tuning feature, otherwise specifies the step size 4
const int  preTune = 0;                //0=OFF, otherwise specifies the step size (20)
const int tuneDutyStep=20;
unsigned char flgSWmode = 3;           //0 for freq, 1 for duty cycle, 2 for tunePhaseStep size, 3 for voltage, 4 for timeOff
signed int tunePhaseStep=40;           //0=disable, phase tune step size 20 (in timer values) 40 for 265 hz
#define phaseCal    300                //phase calibration 50 subtracts 5 degs

#pragma PERSISTENT ( pk )
unsigned int pk=24000000/fTarget;   //counter value matching peak signal
#pragma PERSISTENT ( vSet )
signed char vSet = 3;                  //default to lowest power/voltage setting.
#pragma PERSISTENT ( tOff )
signed char tOff = 2;                  //default to 450ms off
#pragma PERSISTENT ( fmode )
char fmode = 'R';                       //default to rotary field (R=rotary, V=Vert, H=Horz)
char fmode_Act=1;                       //active field mode count
unsigned char tpf;                      //tune phase flag (so phase is only tuned during rotating field mode)
//CALIBRATION VALUES FOR mT LEVELS

/*
// upper LOOP coil voltages (vert field) 265Hz
//                0  1    2    3    4     5   6    7    8     9
uint8_t dcLc[10]={0,0x05,0x08,0x19,0x30,0x31,0x34,0x3c,0x3d,0x3e};      //normal burst voltage (loop)
uint8_t dcLcPB[10]={0,0x10,0x19,0x30,0x48,0x4B,0x60,0x5F,0x6F,0x6F};   //Pre-Burst boost voltage (loop)
uint8_t dcLcB[10]={0,0x10,0x19,0x30,0x40,0x40,0x48,0x4f,0x4F,0x4F};   //Early Burst boost voltage (loop)
// lower SPIRAL coil voltages (horz field)
uint8_t dcSc[10]={0,0x0A,0x0F,0x1F,0x36,0x3C,0x3E,0x4f,0x60,0x5f};      //normal burst voltage (spiral)
uint8_t dcScPB[10]={0,0x20,0x28,0x2E,0x58,0x58,0x6F,0x6F,0x6F,0x6f};   //Pre-Burst boost voltage (spiral)
uint8_t dcScB[10]={0,0x20,0x28,0x2E,0x50,0x50,0x58,0x5f,0x6F,0x6f};   //Early Burst boost voltage (spiral)

// upper LOOP coil voltages (vert field) 1081Hz
//                0  1    2    3    4     5   6    7    8     9
uint8_t dcLc[10]={0,0x12,0x1a,0x30,0x3a,0x4a,0x4f,0x6f,0x6f,0x6f};      //normal burst voltage (loop)
uint8_t dcLcPB[10]={0,0x1b,0x1f,0x30,0x48,0x4f,0x6f,0x6F,0x6F,0x6F};   //Pre-Burst boost voltage (loop)
uint8_t dcLcB[10]={0,0x1f,0x30,0x40,0x48,0x60,0x6f,0x6f,0x6F,0x6F};   //Early Burst boost voltage (loop)
// lower SPIRAL coil voltages (horz field)
uint8_t dcSc[10]={0,0x0A,0x0F,0x1e,0x30,0x3C,0x3E,0x4f,0x60,0x5f};      //normal burst voltage (spiral)
uint8_t dcScPB[10]={0,0x20,0x28,0x28,0x3e,0x58,0x6F,0x6F,0x6F,0x6f};   //Pre-Burst boost voltage (spiral)
uint8_t dcScB[10]={0,0x20,0x28,0x38,0x3e,0x50,0x58,0x6f,0x6F,0x6f};   //Early Burst boost voltage (spiral)
*/
// delay for boost values in ms
uint8_t db[14] ={0, 0, 0, 8,8,8,8,8,8,8,8,8,8};        //db+df (keep total delay in each case the same)
uint8_t df[14] ={16,16,16,8,8,8,8,8,8,8,8,8,8};      //total delay is delay before starting burst measurements (16ms)
// upper LOOP coil voltages (vert field) 265Hz
// Field Levels:    0    0.5  1   1.5   2   2.5   3   3.5   4    4.5   5     mT PEAK
// Setting:         0    1    2    3    4    5    6    7    8     9    A    B
uint8_t dcLcPB[14]={0x00,0x01,0x12,0x19,0x30,0x3d,0x48,0x60,0x5F,0x7b,0x7f,0x7F}; //1.Pre-Burst boost V (loop)
uint8_t dcLcB[14]= {0x00,0x01,0x11,0x20,0x16,0x29,0x39,0x48,0x4f,0x5c,0x70,0x70}; //2.Early Burst boost V (loop)
uint8_t dcLc[14]=  {0x00,0x00,0x05,0x10,0x13,0x21,0x30,0x34,0x3c,0x48,0x4f,0x4f}; //3.normal burst V (loop)
// lower SPIRAL coil voltages (horz field)
uint8_t dcScPB[14]={0x00,0x04,0x20,0x30,0x39,0x50,0x58,0x6F,0x6F,0x7b,0x7f,0x7f}; //1.Pre-Burst boost V (spiral)
uint8_t dcScB[14]= {0x00,0x00,0x20,0x21,0x3a,0x3e,0x58,0x58,0x5f,0x7b,0x79,0x79}; //2.Early Burst boost V (spiral)
uint8_t dcSc[14]=  {0x00,0x01,0x0A,0x13,0x1F,0x34,0x3C,0x3E,0x4f,0x5c,0x6b,0x6b}; //3.normal burst V (spiral)
//*********************************************************************************
//******************************* MAIN ********************************************
int main(void)
{
    WDTCTL = WDTPW | WDTHOLD;   // stop watchdog timer

    //set clock for 24MHz
    // FRAM wait states for the FR5739 are inserted automatically
/*    CSCTL0_H = 0xA5;
    CSCTL1 |= DCORSEL + DCOFSEL0 + DCOFSEL1;   // Set max. DCO setting
    CSCTL2 = SELA_3 + SELS_3 + SELM_3;        // set ACLK = MCLK = DCO
    CSCTL3 = DIVA_0 + DIVS_0 + DIVM_0;        // set all dividers to 0
*/
    //configure for 24 MHz external oscillator PJ.4(XIN), PJ.5(XOUT)
    PJSEL0 |=  BIT5 | BIT4;     //set PJ.5 as xout and PJ.4 as xin
    CSCTL0_H = 0xA5;                            //password
    CSCTL1 |= DCORSEL + DCOFSEL0 + DCOFSEL1;   // Set max. DCO setting
    CSCTL2 = SELA_0 + SELS_0 + SELM_0;        // set all clk sources for XT1
    CSCTL3 = DIVA_0 + DIVS_0 + DIVM_0;        // set all dividers to 0
    CSCTL4 |= XT1DRIVE_3 + XTS + XT1BYPASS;   //highest drive, high freq mode, external clk
    //wait for clock to start
    //clear XT1 and DCO fault flag and interrupt
    do{CSCTL5 &= ~XT1OFFG; SFRIFG1 &= ~OFIFG;}
    while (SFRIFG1 & OFIFG);

    WDTCTL = wdt_start; // start the Watchdog

    // PORT 1  --  7(I2C-SCL) 6(I2C-SDA) 5(en) 4(ADCin) 3(phase in) 2(en) 1(Timer in) 0(Timer Out)
    P1DIR = HBen;     //P1.2,P1.5 en outputs for HB, all else inputs
    //setup TA0 output pin to P1.0
    P1DIR |= BIT0;                       // P1.0 output TA0 output to output ~3KHz timer output
    P1SEL0 |= BIT0;                      // P1.0 options select
    P1SEL0 |= BIT1;                 //Bit 1 is the TA1CLK which is the input for Timer TA1

    // PORT 2 (all outputs)
    // 7(pre) 6(clr) 5(pre) 4(Clr) 3(ADC) 2(LCD reset) 1(sw) 0(sw)
    P2DIR = BIT7 | BIT6 | BIT5 | BIT4 | BIT2;   //outputs
    P2REN |= BIT1 | BIT0;                      //P2.0 and P2.1 pull-up register enable for switches
    P2OUT = BIT7 | BIT6 | BIT5 | BIT4 | BIT1 | BIT0;          //pre,clr high for D flipFlops BITS 0 and 1 are set for pull up resistors
    P2IES |= (bDW+bUP);             //sw: P2.0,1 Hi/Lo edge
    P2IE |= (bDW+bUP);              //sw: P2.0,1 interrupt enabled
    P2IFG &= ~(bDW+bUP);            //sw: P2.0,1 IFG cleared

    // PORT 3 (all outputs)
    // 7(fet) 6(fet) 5(fet) 4(phase in) 3(fet)2(fet) 1(fet) 0(LM317)
    P3DIR = BIT7 | BIT6 | BIT5 | BIT3 | BIT2 | BIT1 | BIT0; //all output except bit 4
    P3OUT = 0x00;   //BIT0 low for LM317 so it starts enabled and reset later (Vbb doubler then works)
    P3OUT &= ~LM317;        //turns on LM317 voltage so half bridge gets powered (to avoid exceeding differential voltage spec of 47V)

    //PORT4  1(SW) 0(NU)
    P4DIR = 0x00 | BIT0;           //all inputs except P4.0
    P4REN |= BIT1;   //P4 pull-up register enable
    P4OUT =  BIT1;   //set for pull up resistors
    P4IES |= BIT1;   //sw: P4.1 Hi/Lo edge
    P4IE |= BIT1;    //sw: P4.1 interrupt enabled
    P4IFG &= ~BIT1;  //sw: P4.1 IFG cleared

    // Configure ADC10
    // By default, REFMSTR=1 => REFCTL is used to configure the internal reference
     while(REFCTL0 & REFGENBUSY);              // If ref generator busy, WAIT
     REFCTL0 |= REFVSEL_2+REFON;               // Select internal ref = 2.5V. Internal Reference ON
     ADC10CTL0 = ADC10SHT_2 + ADC10ON;
     ADC10CTL1 |= ADC10SHP;                    // ADCCLK = MODOSC; sampling timer
     ADC10CTL2 |= ADC10RES;                    // 10-bit conversion results
 //    ADC10IE |= ADC10IE0;                      // Enable ADC conv complete interrupt (routine now uses polling)
     ADC10MCTL0 |= ADC10INCH_1 + ADC10SREF_1;  // A0 ADC input select; Vref=1.5V

    // Configure
    PMMCTL0_H = PMMPW_H;                        // Unlock the PMM registers
    PM5CTL0 &= ~LOCKLPM5;                       // Disable the GPIO power-on default high-impedance mode
                                                 // to activate previously configured port settings
    // load the shift registers to set the DC/DC voltages
    for (xs=0x0080; xs>0; xs=xs>>1) {
        if (0x00 & xs) {P3OUT &= ~BIT1;} else {P3OUT |= BIT1;}  //set up uSR bit
        if (0x00 & xs) {P3OUT &= ~BIT2;} else {P3OUT |= BIT2;}  //set up lSR bit
        __delay_cycles(1); P3OUT |= BIT3; __delay_cycles(1); P3OUT &= ~BIT3; //clock in the bit
    }
    __delay_cycles(1); P3OUT |= BIT3; __delay_cycles(1); P3OUT &= ~BIT3; //last clock to xfer to output register

    delay_ms(3000);                             //3000ms startup delay for power supply power up and cap charging
    //reset half bridge driver power
    P1OUT &= ~HBen;         //Half Bridge OFF
    //P3OUT |= LM317;         //turn off half bridge driver using LM317 (UPDATE: must leave on for voltage inputs higher than about 50V)
    delay_ms(6);            //6ms delay
    P3OUT &= ~LM317;        //turns on LM317 voltage so half bridge gets powered

    //setup hardware clock resonant frequency, output goes to P1.0 (scanning starting point)
    TA0CCR0 = (24000000/fTarget)-1;                     // PWM Period
    TA0CCTL1 = OUTMOD_7;                      // CCR1 reset/set
    TA0CCR1 = TA0CCR0>>1;                     // CCR1 PWM duty cycle (50% is divid by 2)
    TA0CTL = TASSEL__SMCLK | MC__UP | TACLR;  // SMCLK, up mode, clear TAR

    i2cm_start();
    i2cm_stop();
    init_LCD();     //initialize LCD
    startUP_LCD();
    delay_ms(3000);         //3 second startup delay to display LCD startup

    //setup timer for switch debounce
    TB0CCTL0 = CCIE;                          // TACCR0 interrupt enabled
    TB0CCR0 = 1000;
    TB0CTL = TBSSEL__ACLK | MC__UP | TBCLR;     // ACLK, up mode

    //setup interrupt to control the coil On/Off Timing
    // uses P1.6 as input
    TA1CCTL0 = CCIE;                            // TACCR0 interrupt enabled
    TA1CCR0 = offTime2;                       // Period (actual time is defined in the Timer interrupt code)
    TA1CTL = TASSEL__TACLK | MC__CONTINUOUS | TACLR;    // DIV by 1
    __enable_interrupt();
    SFRIE1 |= WDTIE;                            // Enable WDT interrupt

    //this is now done in initialaztion
    //pk= 24000000/fTarget;                  // go directly to the target frequency

    // restore peak value
    TA0CCR0 = pk-1;                           // PWM Period
    TA0CCTL1 = OUTMOD_7;                      // CCR1 reset/set
    TA0CCR1 = TA0CCR0>>1;                     // CCR1 PWM duty cycle (50% is divid by 2)
    TA0CTL = TASSEL__SMCLK | MC__UP | TACLR;  // SMCLK, up mode, clear TAR

    // **************************** MAIN LOOP **********************************
    while(1)
    {
        // ------------------- Update LCD display --------------------------------
         // takes about 1ms per line to update the display
         // 4.2ms for 4 lines (existing use measures 2.2ms, including floating point math)
         // use of subroutines minimized for reduce stack depth when interrupt is called
         //      i2c timing not destroyed by brief interrupt occurrence
        // technically there should be no interrupts occurring during the LCD update (except watchdog)
         //write characters to LCD display
         i2cm_start();
         i2cm_out(0x7A);  //send address with write byte
         i2cm_out(0x80);   //control byte  (no more continuation)
         i2cm_out(0x01);   //Clear Display: Display on: cursor on, blink on

         //Line 1
         //display freq
         freq=12000000/pk;      //compensates for /2: only takes 21us measures to perform this division
         numL2_ascii(freq);    //convert to ASCII char for LCD display (output in pC[] array)
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[4]); //Co bit + DATA byte, digit 4
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[3]); //Co bit + DATA byte, digit 3
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[2]); //Co bit + DATA byte, digit 2
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[1]); //Co bit + DATA byte, digit 1
         i2cm_out(0x80 | 0x40);  i2cm_out('H'); //Co bit + DATA byte, ASCII
         i2cm_out(0x80 | 0x40);  i2cm_out('z'); //Co bit + DATA byte
         //i2cm_out(0x80 | 0x40);  i2cm_out(0x20); //Co bit + DATA byte, Space

         //display duty cycle
         tfloat = TA0CCR1;                      //reassign to allow floating point calculation
         dutyCycle=(tfloat/TA0CCR0)*1000.0;     //40us to perform
         numL2_ascii(dutyCycle);    //convert to ASCII char for LCD display (output in pC[] array)
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[4]); //Co bit + DATA byte, digit 4
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[3]); //Co bit + DATA byte, digit 3
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[2]); //Co bit + DATA byte, digit 2
         i2cm_out(0x80 | 0x40);  i2cm_out('.'); //Co bit + DATA byte, digit decimal point
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[1]); //Co bit + DATA byte, digit 1
         i2cm_out(0x80 | 0x40);  i2cm_out('%'); //Co bit + DATA byte, percent
         i2cm_out(0x80 | 0x40);  i2cm_out(0x20); //Co bit + DATA byte, Space

         //display tunePhaseStep
         if (tunePhaseStep==0){
             i2cm_out(0x80 | 0x40);  i2cm_out('0'); //Co bit + DATA byte, digit 1
         } else if (tunePhaseStep<=20){
             i2cm_out(0x80 | 0x40);  i2cm_out('1'); //Co bit + DATA byte, digit 1
         } else {
             i2cm_out(0x80 | 0x40);  i2cm_out('2'); //Co bit + DATA byte, digit 1
         }
         i2cm_out(0x80 | 0x40);  i2cm_out(0x20);  //Co bit + DATA byte, Space

         //display vSet
         //numL2_ascii(vSet);    //convert to ASCII char for LCD display (output in pC[] array)
         char2_HEX(vSet);    //convert to ASCII char for LCD display (output in pC[] array)
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[1]); //Co bit + DATA byte, digit 1
         i2cm_out(0x80 | 0x40);  i2cm_out(0x20); //Co bit + DATA byte, Space

         //display tOff
         numL2_ascii(tOff);    //convert to ASCII char for LCD display (output in pC[] array)
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[1]); //Co bit + DATA byte, digit 1
         i2cm_out(0x80 | 0x40);  i2cm_out(0x20); //Co bit + DATA byte, Space

         //display field mode (fmode)
         i2cm_out(0x80 | 0x40);  i2cm_out(fmode);  //Co bit + DATA byte, Field mode
 /*
         //display SW mode
         if (flgSWmode==0) {
             i2cm_out(0x80 | 0x40);  i2cm_out('F'); //Co bit + DATA byte, frequency adjust mode
         } else if (flgSWmode==1){
             i2cm_out(0x80 | 0x40);  i2cm_out('D'); //Co bit + DATA byte, duty cycle adjust mode
         } else if (flgSWmode==2){
             i2cm_out(0x80 | 0x40);  i2cm_out('P'); //Co bit + DATA byte, phase step size adjust mode
         } else if (flgSWmode==3){
             i2cm_out(0x80 | 0x40);  i2cm_out('V');  //Co bit + DATA byte, Voltage adjust mode
         } else {
             i2cm_out(0x80 | 0x40);  i2cm_out('T'); //Co bit + DATA byte, tOff adjust mode
         }
*/

         //line 2
 /*        //display voltage upper setting
         char2_HEX(dcLc[vSet]);    //convert to ASCII char for LCD display (output in pC[] array)
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[2]); //Co bit + DATA byte, digit 1
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[1]); //Co bit + DATA byte, digit 1
         i2cm_out(0x80 | 0x40);  i2cm_out(0x20); //Co bit + DATA byte, Space
         //display voltage Upper Boost setting
         char2_HEX(dcLcB[vSet]);    //convert to ASCII char for LCD display (output in pC[] array)
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[2]); //Co bit + DATA byte, digit 1
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[1]); //Co bit + DATA byte, digit 1
         i2cm_out(0x80 | 0x40);  i2cm_out(0x20); //Co bit + DATA byte, Space
         //display voltage lower setting
         char2_HEX(dcSc[vSet]);    //convert to ASCII char for LCD display (output in pC[] array)
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[2]); //Co bit + DATA byte, digit 1
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[1]); //Co bit + DATA byte, digit 1
         i2cm_out(0x80 | 0x40);  i2cm_out(0x20); //Co bit + DATA byte, Space
         //display v lower Boost setting
         char2_HEX(dcScB[vSet]);    //convert to ASCII char for LCD display (output in pC[] array)
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[2]); //Co bit + DATA byte, digit 1
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[1]); //Co bit + DATA byte, digit 1
         i2cm_out(0x80 | 0x40);  i2cm_out(0x20); //Co bit + DATA byte, Space
 */
         i2cm_out(0x80 | 0x40);  i2cm_out('F');
         i2cm_out(0x80 | 0x40);  i2cm_out('r');
         i2cm_out(0x80 | 0x40);  i2cm_out('e');
         i2cm_out(0x80 | 0x40);  i2cm_out('q');
         i2cm_out(0x80 | 0x40);  i2cm_out(' ');
         i2cm_out(0x80 | 0x40);  i2cm_out(' ');
         i2cm_out(0x80 | 0x40);  i2cm_out(' ');
         i2cm_out(0x80 | 0x40);  i2cm_out('D');
         i2cm_out(0x80 | 0x40);  i2cm_out('C');
         i2cm_out(0x80 | 0x40);  i2cm_out('%');
         i2cm_out(0x80 | 0x40);  i2cm_out(' ');
         i2cm_out(0x80 | 0x40);  i2cm_out(' ');

         i2cm_out(0x80 | 0x40);  i2cm_out('P'); //Co bit + DATA byte, ascii (phase step)
         i2cm_out(0x80 | 0x40);  i2cm_out('S'); //Co bit + DATA byte, ascii
         i2cm_out(0x80 | 0x40);  i2cm_out(' '); //Co bit + DATA byte, ascii
         i2cm_out(0x80 | 0x40);  i2cm_out('V'); //Co bit + DATA byte, ascii (voltage)
         i2cm_out(0x80 | 0x40);  i2cm_out(' '); //Co bit + DATA byte, ascii
         i2cm_out(0x80 | 0x40);  i2cm_out('T'); //Co bit + DATA byte, ascii (off time)
         i2cm_out(0x80 | 0x40);  i2cm_out(' '); //Co bit + DATA byte, ascii
         i2cm_out(0x80 | 0x40);  i2cm_out(' '); //Co bit + DATA byte, ascii


         //line 3
         //display Spiral voltage mid burst
         numL2_ascii(V2a);    //convert to ASCII char for LCD display (output in pC[] array)
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[4]); //Co bit + DATA byte, digit 4
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[3]); //Co bit + DATA byte, digit 3
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[2]); //Co bit + DATA byte, digit 2
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[1]); //Co bit + DATA byte, digit 1
         i2cm_out(0x80 | 0x40);  i2cm_out('-');  //Co bit + DATA byte
         //display Spiral voltage late burst
         numL2_ascii(V2b);    //convert to ASCII char for LCD display (output in pC[] array)
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[4]); //Co bit + DATA byte, digit 4
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[3]); //Co bit + DATA byte, digit 3
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[2]);  //Co bit + DATA byte, digit 2
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[1]); //Co bit + DATA byte, digit 1
         i2cm_out(0x80 | 0x40);  i2cm_out(' '); //Co bit + DATA byte
         //display Loop voltage mid burst
         numL2_ascii(V1a);    //convert to ASCII char for LCD display (output in pC[] array)
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[4]); //Co bit + DATA byte, digit 4
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[3]); //Co bit + DATA byte, digit 3
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[2]);  //Co bit + DATA byte, digit 2
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[1]); //Co bit + DATA byte, digit 1
         i2cm_out(0x80 | 0x40);  i2cm_out('-'); //Co bit + DATA byte
         //display Loop voltage late burst
         numL2_ascii(V1b);    //convert to ASCII char for LCD display (output in pC[] array)
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[4]); //Co bit + DATA byte, digit 4
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[3]); //Co bit + DATA byte, digit 3
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[2]);  //Co bit + DATA byte, digit 2
         i2cm_out(0x80 | 0x40);  i2cm_out(pC[1]); //Co bit + DATA byte, digit 1
         i2cm_out(0x80 | 0x40);  i2cm_out(' '); //Co bit + DATA byte

        //display phase values (early burst)
         tfloat = phf1-phs1;
         phase1 = (tfloat/phCyc1)*1800.0;
         phase1=phase1+phaseCal;        //correct/calibrate measurement by x deg
        numL2_ascii(phase1);    //convert to ASCII char for LCD display (output in pC[] array)
        i2cm_out(0x80 | 0x40);  i2cm_out(pC[4]); //Co bit + DATA byte, digit 4
        i2cm_out(0x80 | 0x40);  i2cm_out(pC[3]);  //Co bit + DATA byte, digit 3
        i2cm_out(0x80 | 0x40);  i2cm_out(pC[2]); //Co bit + DATA byte, digit 2
        i2cm_out(0x80 | 0x40);  i2cm_out('.'); //Co bit + DATA byte, Decimal
        i2cm_out(0x80 | 0x40);  i2cm_out(pC[1]); //Co bit + DATA byte, digit 1
        i2cm_out(0x80 | 0x40);  i2cm_out('-'); //Co bit + DATA byte


        //display phase values (mid burst)
        tfloat = phf2-phs2;
        phase2 = (tfloat/phCyc2)*1800.0;
        phase2=phase2+phaseCal;        //correct/calibrate measurement by x deg
        numL2_ascii(phase2);    //convert to ASCII char for LCD display (output in pC[] array)
        i2cm_out(0x80 | 0x40);  i2cm_out(pC[4]); //Co bit + DATA byte, digit 4
        i2cm_out(0x80 | 0x40);  i2cm_out(pC[3]);  //Co bit + DATA byte, digit 3
        i2cm_out(0x80 | 0x40);  i2cm_out(pC[2]); //Co bit + DATA byte, digit 2
        i2cm_out(0x80 | 0x40);  i2cm_out('.'); //Co bit + DATA byte
        i2cm_out(0x80 | 0x40);  i2cm_out(pC[1]);  //Co bit + DATA byte, digit 1
        i2cm_out(0x80 | 0x40);  i2cm_out('-'); //Co bit + DATA byte


        //display phase values (late burst)
        tfloat = phf3-phs3;
        phase3 = (tfloat/phCyc3)*1800.0;
        phase3=phase3+phaseCal;        //correct/calibrate measurement by x deg
        numL2_ascii(phase3);    //convert to ASCII char for LCD display (output in pC[] array)
        i2cm_out(0x80 | 0x40);  i2cm_out(pC[4]); //Co bit + DATA byte, digit 4
        i2cm_out(0x80 | 0x40);  i2cm_out(pC[3]); //Co bit + DATA byte, digit 3
        i2cm_out(0x80 | 0x40);  i2cm_out(pC[2]); //Co bit + DATA byte, digit 2
        i2cm_out(0x80 | 0x40);  i2cm_out('.'); //Co bit + DATA byte
        i2cm_out(0x80 | 0x40);  i2cm_out(pC[1]); //Co bit + DATA byte, digit 1
        i2cm_out(0x80 | 0x40);  i2cm_out('d'); //Co bit + DATA byte
        i2cm_out(0x80 | 0x40);  i2cm_out('e'); //Co bit + DATA byte
        //last one
        i2cm_out(0x40);  i2cm_out('g'); // DATA byte


         i2cm_stop();

         delay_ms(1);
         //reposition cursor
         i2cm_start();
         i2cm_out(0x7A);  //send address with write byte
             //frequency adjust mode
             i2cm_out(0x80);  i2cm_out(0x14);   //control byte + shift right
             i2cm_out(0x80);  i2cm_out(0x14);   //control byte + shift right
             i2cm_out(0x80);  i2cm_out(0x14);   //control byte + shift right
         if (flgSWmode>=1){   //duty cycle adjust mode
             i2cm_out(0x80);  i2cm_out(0x14);   //control byte + shift right
             i2cm_out(0x80);  i2cm_out(0x14);   //control byte + shift right
             i2cm_out(0x80);  i2cm_out(0x14);   //control byte + shift right
             i2cm_out(0x80);  i2cm_out(0x14);   //control byte + shift right
             i2cm_out(0x80);  i2cm_out(0x14);   //control byte + shift right
             i2cm_out(0x80);  i2cm_out(0x14);   //control byte + shift right
             i2cm_out(0x80);  i2cm_out(0x14);   //control byte + shift right
             i2cm_out(0x80);  i2cm_out(0x14);   //control byte + shift right
         }
         if (flgSWmode>=2){   //phase step size adjust mode
             i2cm_out(0x80);  i2cm_out(0x14);   //control byte + shift right
             i2cm_out(0x80);  i2cm_out(0x14);   //control byte + shift right
         }
         if (flgSWmode>=3){   //voltage adjust mode
             i2cm_out(0x80);  i2cm_out(0x14);   //control byte + shift right
             i2cm_out(0x80);  i2cm_out(0x14);   //control byte + shift right
         }
         if (flgSWmode>=4){   //Toff adjust mode
             i2cm_out(0x80);  i2cm_out(0x14);   //control byte + shift right
             i2cm_out(0x80);  i2cm_out(0x14);   //control byte + shift right
         }
         if (flgSWmode>=5){   //Toff adjust mode
             i2cm_out(0x80);  i2cm_out(0x14);   //control byte + shift right
             i2cm_out(0x80);  i2cm_out(0x14);   //control byte + shift right
         }

         i2cm_stop();



         delay_ms(100);         //delay before boost and timing adjust

         // set the boost voltage values
         // load the shift registers to set the DC/DC voltages
         for (xs=0x0080; xs>0; xs=xs>>1) {
             if (dcLcPB[vSet] & xs) {P3OUT &= ~BIT1;} else {P3OUT |= BIT1;}  //set up uSR bit
             if (dcScPB[vSet] & xs) {P3OUT &= ~BIT2;} else {P3OUT |= BIT2;}  //set up lSR bit
             __delay_cycles(1); P3OUT |= BIT3; __delay_cycles(1); P3OUT &= ~BIT3; //clock in the bit
         }
         __delay_cycles(1); P3OUT |= BIT3; __delay_cycles(1); P3OUT &= ~BIT3; //last clock to xfer to output register

         //ensure the phase alignment is correct/constant
         //note the timing of the reset also affects which on leads, hence best to wait for known condition before adjusting the timing
         // for example reversing the leads from the RC resonant cct, to the controller will cause a 180 deg phase shift
         while ((TA0R<1000)|(TA0R>1100));
         // P2.5(pre), P2.6(clr) and LCD reset
         // P2.7(pre), P2.4(clr)
         P2OUT &= ~BIT5;     //pulse low for preset
         P2OUT &= ~BIT7;     //pulse low for preset (CH 2 leads)
         //P2OUT &= ~BIT4;     //pulse low for preset (CH1 leads)
         P2OUT |= BIT5 + BIT6;
         P2OUT |= BIT7 + BIT4;


       // __bis_SR_register(LPM4_bits | GIE);     // Enter LPM4, enable interrupts
        __bis_SR_register(LPM3_bits | GIE);     // Enter LPM3, enable interrupts
    }
    return 0;
}




//********************* Interrupt Routines ********************************
// ***************************************************************


// **********************************************************
// *******************   WDT INTERRUPT **********************
// Watchdog Timer interrupt service routine
// arrives here when HB needs to be sent.
//  Note that VLO(not very stable) is used for watchdog so that the HBs are not aligned in time between different units
#pragma vector=WDT_VECTOR
__interrupt void WDT_ISR(void)
{
    //verify program working by checking cntMain is incrementing
    if (cntMain==cntMainPrev) {
        //MCU hung, so MCU needs reboot
        PMMCTL0 |= PMMSWBOR;    //cause a software brown out reset
    } else {
        cntMainPrev=cntMain;
    }
}//WD interrupt

// ***************** PORT2 INTERRUPT ***************************
// Port 2 interrupt service routine
// button P2.0(up) P2.1(down)
#pragma vector=PORT2_VECTOR
__interrupt void Port_2(void)
{
    switch(flgSWmode) {
      case 0:
        //check input switches (used for frequency adjust)
         if ((P2IN&bUP)==0) {     // if button pressed
             //adjust frequency, setup hardware clock, output goes to P1.1
             pk=pk-tuneManStep;
             TA0CCR0 = pk;                            // PWM Period
             TA0CCTL1 = OUTMOD_7;                      // CCR1 reset/set
             TA0CCR1 = TA0CCR0>>1;                     // CCR1 PWM duty cycle (50% is divid by 2)
             TA0CTL = TASSEL__SMCLK | MC__UP | TACLR;  // SMCLK, up mode, clear TAR
         }
         if ((P2IN&bDW)==0) {     // if button pressed
             //adjust frequency, setup hardware clock, output goes to P1.1
             pk=pk+tuneManStep;
             TA0CCR0 = pk;                            // PWM Period
             TA0CCTL1 = OUTMOD_7;                      // CCR1 reset/set
             TA0CCR1 = TA0CCR0>>1;                     // CCR1 PWM duty cycle (50% is divid by 2)
             TA0CTL = TASSEL__SMCLK | MC__UP | TACLR;  // SMCLK, up mode, clear TAR
         }
         break;
      case 1:
        //check input switches (used for duty cycle adjust)
         if ((P2IN&bUP)==0) {     // if button pressed
             TA0CCTL1 = OUTMOD_7;                      // CCR1 reset/set
             if (TA0CCR1>tuneDutyStep) {       // defines bounds
                 TA0CCR1=TA0CCR1-tuneDutyStep;
             }
             TA0CTL = TASSEL__SMCLK | MC__UP | TACLR;  // SMCLK, up mode, clear TAR
         }
         if ((P2IN&bDW)==0) {     // if button pressed
             TA0CCTL1 = OUTMOD_7;                      // CCR1 reset/set
             if (TA0CCR1<(pk+tuneDutyStep)) {       // defines bounds
                 TA0CCR1=TA0CCR1+tuneDutyStep;
             }
             TA0CTL = TASSEL__SMCLK | MC__UP | TACLR;  // SMCLK, up mode, clear TAR
         }
         break;
      case 2:    //tunePhaseStep size
          if ((P2IN&bDW)==0) {     // if button pressed
              //adjust tunePhaseStep size
              tunePhaseStep=tunePhaseStep-20;
              if (tunePhaseStep<0) tunePhaseStep=40;
          }
          if ((P2IN&bUP)==0) {     // if button pressed
              //adjust tunePhaseStep size
              tunePhaseStep=tunePhaseStep+20;
              if (tunePhaseStep>40) tunePhaseStep=0;
          }
          break;
      case 3:    //voltage setting
          if ((P2IN&bUP)==0) {     // if button pressed
              vSet++;
              if (vSet>10) vSet=0;
          }
          if ((P2IN&bDW)==0) {     // if button pressed
              vSet--;
              if (vSet<0) vSet=10;
          }
          break;
      case 4:    //tOff time
          if ((P2IN&bUP)==0) {     // if button pressed
              tOff++;
              if (tOff>7) tOff=1;
          }
          if ((P2IN&bDW)==0) {     // if button pressed
              tOff--;
              if (tOff<1) tOff=7;
          }
          break;
      case 5:    //Field mode
          if ((P2IN&bUP)==0) {     // if button pressed
              if (fmode=='R') fmode='V';
              else if (fmode=='V') fmode='H';
              else if (fmode=='H') fmode='1';
              else if (fmode=='1') fmode='2';
              else if (fmode=='2') fmode='R';
          }
          if ((P2IN&bDW)==0) {     // if button pressed
              if (fmode=='R') fmode='2';
              else if (fmode=='V') fmode='R';
              else if (fmode=='H') fmode='V';
              else if (fmode=='1') fmode='H';
              else if (fmode=='2') fmode='1';
          }
          break;
      default:
          flgSWmode=3;
          break;
    }//sw

    P2IE &= ~(bDW+bUP);     //sw: P2.0,1 interrupt DISABLED (for debounce)
    P2IFG &= ~(bDW+bUP);    // Clear P2.0,1 IFG
    TB0CCR0 += 2000;  //set timer for debounce (about 60ms)
    TB0CTL = TBSSEL__ACLK | MC__UP;     // ACLK, up modeG
}

// ***************** PORT2 INTERRUPT ***************************
// Port 4 interrupt service routine
// MODE button P4.1
#pragma vector=PORT4_VECTOR
__interrupt void Port_4(void)
{
    //MODE button has been depressed
    // note interrupt is rising edge but switch can still bounce
    //test for mode switch depressed to toggle the switch mode
    if ((P4IN&BIT1)==0){
        //toggle the mode
        flgSWmode++;  //increment mode
        //if (flgSWmode>5) flgSWmode=3;  //causes mode to loop back but skips freq, duty cycle, phase step tuning (not required to tune)
        if (flgSWmode>5) flgSWmode=0;  //all tuning modes
       // init_LCD();     //initialize LCD
    }
    P4IE &= ~BIT1;     //sw: P4.1 interrupt DISABLED (for debounce)
    P4IFG &= ~BIT1;    // Clear P4.1 IFG
    TB0CCR0 += 2000;  //set timer for debounce (about 60ms)
    TB0CTL = TBSSEL__ACLK | MC__UP;     // ACLK, up mode
}

// ***************** TIMER B0 INTERRUPT ***************************
// Timer B0 interrupt service routine
// used for switch debounce
#pragma vector = TIMER0_B0_VECTOR
__interrupt void Timer_B (void)
{
    P2IFG &= ~(bDW+bUP);    //sw: P2.0,1 IFG cleared
    P2IE |= (bDW+bUP);      //sw: P2.0,1 interrupt enabled

    P4IFG &= ~BIT1;         // Clear P4.1 IFG
    P4IE |= BIT1;           //sw: P4.1 interrupt enabled

    TB0CTL = TBSSEL__ACLK | MC__STOP;             // ACLK, Timer STOPPED

    //button was depressed, so allow the LCD display to update
    __bic_SR_register_on_exit(CPUOFF);       // Equivalent to exit in LPM0 with cpu clk  (interrupts enabled-not cleared)

}


// ***************** Timer A1 INTERRUPT ***************************
// Timer A1 interrupt service routine
// used for delays
#pragma vector = TIMER1_A0_VECTOR
__interrupt void Timer1_A0_ISR(void)
{
    //coil sequence
    switch(flg) {
        // Coil1 sequence
        case 1:
            // set the boost voltage values
            // load the shift registers to set the DC/DC voltages
            for (xs=0x0080; xs>0; xs=xs>>1) {
                if (dcLcB[vSet] & xs) {P3OUT &= ~BIT1;} else {P3OUT |= BIT1;}  //set up uSR bit
                if (dcScB[vSet] & xs) {P3OUT &= ~BIT2;} else {P3OUT |= BIT2;}  //set up lSR bit
                __delay_cycles(1); P3OUT |= BIT3; __delay_cycles(1); P3OUT &= ~BIT3; //clock in the bit
            }
            __delay_cycles(1); P3OUT |= BIT3; __delay_cycles(1); P3OUT &= ~BIT3; //last clock to xfer to output register

            TA1CCR0 += onTime;      //typically 40ms

            if (vSet) {
                //set what mode we are in to turn on the correct fields (tpf is tune phase flag)
                if (fmode=='R') {tpf=1; P1OUT |= HBen;}          //EN half bridge, turn ON both fields
                if (fmode=='V') {tpf=0; P1OUT |= FBenLoop;}      //if only vertical field
                if (fmode=='H') {tpf=0; P1OUT |= FBenSpiral;}    //if only horizontal field
                if (fmode=='1'){
                    fmode_Act++;
                    if (fmode_Act>4) {fmode_Act=1;}  //loop count
                    if (fmode_Act==1) {tpf=1; P1OUT |= HBen;}          //EN half bridge, turn ON both fields
                    if (fmode_Act==2) {tpf=0; P1OUT |= FBenLoop;}      //if only vertical field
                    if (fmode_Act==3) {tpf=1; P1OUT |= HBen;}          //EN half bridge, turn ON both fields
                    if (fmode_Act==4) {tpf=0; P1OUT |= FBenSpiral;}    //if only horizontal field
                }
                if (fmode=='2'){
                    fmode_Act++;
                    if (fmode_Act>3) {fmode_Act=1;}  //loop count
                    if (fmode_Act==1) {tpf=1; P1OUT |= HBen;}          //EN half bridge, turn ON both fields
                    if (fmode_Act==2) {tpf=0; P1OUT |= FBenLoop;}      //if only vertical field
                    if (fmode_Act==3) {tpf=0; P1OUT |= FBenSpiral;}    //if only horizontal field
                }

            } else {
                //vset=0 meaning disable both sets of drivers
                //  technically means just don't enable anything
            }

            // configure boost disable time based on vSet setting (keep total delay in each case the same)
            for (xt= db[vSet]; xt > 0; xt--) {__delay_cycles(cM*1000);}   //delay for boost disable
            // load the shift registers to set the DC/DC voltages
            for (xs=0x0080; xs>0; xs=xs>>1) {
                if (dcLc[vSet] & xs) {P3OUT &= ~BIT1;} else {P3OUT |= BIT1;}  //set up uSR bit
                if (dcSc[vSet] & xs) {P3OUT &= ~BIT2;} else {P3OUT |= BIT2;}  //set up lSR bit
                __delay_cycles(1); P3OUT |= BIT3; __delay_cycles(1); P3OUT &= ~BIT3; //clock in the bit
            }
            __delay_cycles(1); P3OUT |= BIT3; __delay_cycles(1); P3OUT &= ~BIT3; //last clock to xfer to output register
            for (xt= df[vSet]; xt > 0; xt--) {__delay_cycles(cM*1000);}   //enough delay for burst to substantial start

            //Total time for both coils ADC 2.7ms measured (4clk sample time, 256 averaging, no sleep mode)
           //measure 2nd coil pair
            //ADC10CLK ~5MHz(0.2us) : conversion time ~ 2.5us for 12 cycles.
            ADC10CTL0 &= ~ADC10ENC;               //disable ADC conversion, to allow changing channel in register
            ADC10CTL0 = ADC10SHT_0 | ADC10ON;       // ADCON, S&H=4 ADC clks
            ADC10MCTL0 = ADC10INCH_4 | ADC10SREF_1; // CH2=A4, Channel A4 or A6  (Vref)
            Vt2a=0;
            P3OUT |= BIT7;  //set high (for measuring conversion time only with scope)
            for (b = 128; b > 0; b--) {
                ADC10CTL0 |= ADC10ENC + ADC10SC;         // Sampling and conversion start
                while ((ADC10IFG&ADC10IFG0)==0);       //WAIT for flag set
                resultADC=ADC10MEM0;        //read result also clears flag
                // __bis_SR_register(LPM1_bits | GIE); // Enter LPM3(ADC optional), enable interrupts
                Vt2a= Vt2a + resultADC;             // save previous ADC reading (2bytes)
            }
            V2a=Vt2a>>7;       //divid by 128 for averaging

            //measure 1st coil pair detected voltage
            ADC10CTL0 &= ~ADC10ENC;               //disable ADC conversion, to allow changing channel in register
            ADC10CTL0 = ADC10SHT_0 | ADC10ON;       // ADCON, S&H=4 ADC clks
            ADC10MCTL0 = ADC10INCH_6 | ADC10SREF_1; // CH1=A6, Channel A4 or A6  (Vref)
            Vt1a=0;
            // measures 180us for 8 measurements, so 22 us per measurement -if measure for 3ms (128 measurements * 22us=3ms)
            for (b = 128; b > 0; b--) {
                ADC10CTL0 |= ADC10ENC + ADC10SC;         // Sampling and conversion start
                while ((ADC10IFG&ADC10IFG0)==0);       //WAIT for flag set
                resultADC=ADC10MEM0;        //read result also clears flag
                // __bis_SR_register(LPM3_bits | GIE); // Enter LPM3(ADC optional), enable interrupts
                Vt1a= Vt1a + resultADC;             // save previous ADC reading (2bytes)
            }
            V1a=Vt1a>>7;                  //divid by 128 for averaging
            ADC10CTL0 &= ~ADC10ENC;     //disable ADC conversion
            P3OUT &= ~BIT7;           //set low (for measuring conversion time only with scope)


            //next perform phase measurement near beginning of burst
            //but only perform if both detected voltages are above a value of 100
            if ((V1a>80)&(V2a>80)) {
                //wait for rising edge of Spiral Coil Phase
                if ((sPIN)>0) {while ((sPIN)>0);}  //if pin is high, then wait for pin to go low
                while ((sPIN)==0);                  //wait for pin to go high
                phs1=TA0R;                          //initialize
                P4OUT |= BIT0;
                while (!(lPIN));                    //wait for Loop Coil PHase to go high
                phf1=TA0R;
                if (phf1<phs1){phf1=phf1+TA0CCR0;}  //fix wrap around
                //continue time measurement until Spiral Coil Phase drops again
                while ((sPIN)>0);                   //wait for CH1 to drop again
                phCyc1=TA0R;
                if (phCyc1<phf1){phCyc1=phCyc1+TA0CCR0;}  //fix wrap around
                P4OUT &= ~BIT0;
            }//if detected voltages OK

            for (xt= 6; xt > 0; xt--) {__delay_cycles(cM*1000);}   //enough delay for
            //next perform phase measurement near middle of burst
            //but only perform if both detected voltages are above a value of 100
            if ((V1a>100)&(V2a>100)) {
                //wait for rising edge of Spiral Coil Phase
                if ((sPIN)>0) {while ((sPIN)>0);}  //if pin is high, then wait for pin to go low
                while ((sPIN)==0);                  //wait for pin to go high
                phs2=TA0R;                          //initialize
                P4OUT |= BIT0;
                while (!(lPIN));                    //wait for Loop Coil PHase to go high
                phf2=TA0R;
                if (phf2<phs2){phf2=phf2+TA0CCR0;}  //fix wrap around
                //continue time measurement until Spiral Coil Phase drops again
                while ((sPIN)>0);                   //wait for CH1 to drop again
                phCyc2=TA0R;
                if (phCyc2<phf2){phCyc2=phCyc2+TA0CCR0;}  //fix wrap around
                P4OUT &= ~BIT0;
            }//if detected voltages OK

            // this delay must be adjusted to match the burst time.
            delay_ms(3);

            //Total time for both coils ADC 2.7ms measured (4clk sample time, 256 averaging, no sleep mode)
           //measure 2nd coil pair
            //ADC10CLK ~5MHz(0.2us) : conversion time ~ 2.5us for 12 cycles.
            ADC10CTL0 &= ~ADC10ENC;               //disable ADC conversion, to allow changing channel in register
            ADC10CTL0 = ADC10SHT_0 | ADC10ON;       // ADCON, S&H=4 ADC clks
            ADC10MCTL0 = ADC10INCH_4 | ADC10SREF_1; // CH2=A4, Channel A4 or A6  (Vref)
            Vt2b=0;
            P3OUT |= BIT7;  //set high (for measuring conversion time only with scope)
            for (b = 128; b > 0; b--) {
                ADC10CTL0 |= ADC10ENC + ADC10SC;         // Sampling and conversion start
                while ((ADC10IFG&ADC10IFG0)==0);       //WAIT for flag set
                resultADC=ADC10MEM0;        //read result also clears flag
                // __bis_SR_register(LPM1_bits | GIE); // Enter LPM3(ADC optional), enable interrupts
                Vt2b= Vt2b + resultADC;             // save previous ADC reading (2bytes)
            }
            V2b=Vt2b>>7;       //divid by 128 for averaging

            //measure 1st coil pair detected voltage
            ADC10CTL0 &= ~ADC10ENC;               //disable ADC conversion, to allow changing channel in register
            ADC10CTL0 = ADC10SHT_0 | ADC10ON;       // ADCON, S&H=4 ADC clks
            ADC10MCTL0 = ADC10INCH_6 | ADC10SREF_1; // CH1=A6, Channel A4 or A6  (Vref)
            Vt1b=0;
            // measures 180us for 8 measurements, so 22 us per measurement -if measure for 3ms (128 measurements * 22us=3ms)
            for (b = 128; b > 0; b--) {
                ADC10CTL0 |= ADC10ENC + ADC10SC;         // Sampling and conversion start
                while ((ADC10IFG&ADC10IFG0)==0);       //WAIT for flag set
                resultADC=ADC10MEM0;        //read result also clears flag
                // __bis_SR_register(LPM3_bits | GIE); // Enter LPM3(ADC optional), enable interrupts
                Vt1b= Vt1b + resultADC;             // save previous ADC reading (2bytes)
            }
            V1b=Vt1b>>7;                  //divid by 128 for averaging
            ADC10CTL0 &= ~ADC10ENC;     //disable ADC conversion
            P3OUT &= ~BIT7;           //set low (for measuring conversion time only with scope)

            //next perform phase measurement
            //but only perform if both detected voltages are above a value of 100
            if ((V1b>80)&(V2b>80)) {
                //wait for rising edge of Spiral Coil Phase
                if ((sPIN)>0) {while ((sPIN)>0);}  //if pin is high, then wait for pin to go low
                while ((sPIN)==0);                  //wait for pin to go high
                phs3=TA0R;                          //initialize
                P4OUT |= BIT0;
                while (!(lPIN));                    //wait for Loop Coil PHase to go high
                phf3=TA0R;
                if (phf3<phs3){phf3=phf3+TA0CCR0;}  //fix wrap around
                //continue time measurement until Spiral Coil Phase drops again
                while ((sPIN)>0);                   //wait for CH1 to drop again
                phCyc3=TA0R;
                if (phCyc3<phf3){phCyc3=phCyc3+TA0CCR0;}  //fix wrap around
                P4OUT &= ~BIT0;
            }//if detected voltages OK


            cntMain++;              //increment in order to verify program working.
            flg=2;                  //setup for next phase
            break;

            case 2:
            switch(tOff) {
                case 1:
                    TA1CCR0 += offTime1; break;
                case 2:
                    TA1CCR0 += offTime2; break;
                case 3:
                    TA1CCR0 += offTime3; break;
                case 4:
                    TA1CCR0 += offTime4; break;
                case 5:
                    TA1CCR0 += offTime5; break;
                case 6:
                    TA1CCR0 += offTime6; break;
                case 7:
                    TA1CCR0 += offTime7; break;
                default:
                    TA1CCR0 += offTime2; break;
            }//sw tOff

            P1OUT &= ~HBen;                         //Half Bridge OFF
            flg=1;                                  //setup for next phase

            //add misc code for off time so it does not interfere with timing


            //tune the phase only if both fields are on
            if (tpf==1) {
                //tune the phase (only adjust if the voltages are high enough)
                if ((V1b>80)&(V2b>80)&(tunePhaseStep>0)) {
                    //tune the phase
                    if (phase2>900) {
                        TA0CCTL1 = OUTMOD_7;                      // CCR1 reset/set
                        TA0CCR1 = TA0CCR1-tunePhaseStep;           // CCR1 PWM duty cycle (50% is divid by 2)
                        TA0CTL = TASSEL__SMCLK | MC__UP | TACLR;  // SMCLK, up mode, clear TAR
                    }
                    if (phase2<900) {
                        TA0CCTL1 = OUTMOD_7;                      // CCR1 reset/set
                        TA0CCR1 = TA0CCR1+tunePhaseStep;           // CCR1 PWM duty cycle (50% is divid by 2)
                        TA0CTL = TASSEL__SMCLK | MC__UP | TACLR;  // SMCLK, up mode, clear TAR
                    }
                }//if
            }//if in Rotary mode (both fields on)

            //allow the LCD display to update during the OFF time
            __bic_SR_register_on_exit(CPUOFF);       // Equivalent to exit in LPM0 with cpu clk  (interrupts enabled-not cleared)

            break;

        default:
            flg=1;
            break;
    }//sw

    //returns to where it was (ie main or sleep) UNLESS specified otherwise (LCD update)
}//timer interrupt



// **********************************************************
// *******************  ADC INTERRUPT **********************
// ADC10 interrupt service routine
#pragma vector=ADC10_VECTOR
__interrupt void ADC10_ISR(void)
{
  switch(__even_in_range(ADC10IV,12))
  {
    case  0: break;                          // No interrupt
    case  2: break;                          // conversion result overflow
    case  4: break;                          // conversion time overflow
    case  6: break;                          // ADC10HI
    case  8: break;                          // ADC10LO
    case 10: break;                          // ADC10IN
    case 12: resultADC = ADC10MEM0;
             __bic_SR_register_on_exit(CPUOFF);
             break;                          // Clear CPUOFF bit from 0(SR)
    default: break;
  }
}




// ***************************** Subroutines *******************************

//LCD startup message
void startUP_LCD(void){
    //write characters to LCD display
    i2cm_start();
      i2cm_out(0x7A);  //send address with write byte
      i2cm_out(0x80);   //control byte  (no more continuation)
      i2cm_out(0x01);   //Clear Display: Display on: cursor on, blink on

      i2cm_out(0x80 | 0x40);  i2cm_out('N'); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out('e');  //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out('s'); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out('t'); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out('r');  //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out('o');  //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out('n'); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out('i'); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out('c'); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out('s'); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out(' ');  //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out(' '); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out(' '); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out(' '); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out(' '); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out(' '); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out(' '); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out(' '); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out(' ');  //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out(' '); //Co bit + DATA byte, ASCII
      //line 2
      i2cm_out(0x80 | 0x40);  i2cm_out('R');  //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out('o'); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out('t'); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out('a'); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out('r'); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out('y'); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out(' '); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out('E'); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out('x'); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out('c');  //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out('i'); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out('t');  //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out('e'); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out('r'); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out(' '); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out(' '); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out(' '); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out(' '); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out(' '); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out(' '); //Co bit + DATA byte, ASCII

      //line 3
      i2cm_out(0x80 | 0x40);  i2cm_out('U'); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out('R'); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out('E'); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out('B'); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out('S'); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out('-'); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out('0'); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out('3'); //Co bit + DATA byte, ASCII
      i2cm_out(0x80 | 0x40);  i2cm_out('b'); //Co bit + DATA byte, ASCII

      i2cm_out(0x40);  i2cm_out(' ');  //last DATA byte

    i2cm_stop();
}




// ---------------------- ASCII characters for LCD display ----------------------------
// Output is in char array pC[]
void numL2_ascii(int32_t vp){
    t32=vp;

    // do 10,000
    pC[5]=0;
    while (t32>=10000){
        t32 = t32 - 10000;
        if (t32>=0) pC[5]++;
    }
    pC[5]=pC[5]+0x30; //convert to ascii by adding 0x30

    // do 1,000
    pC[4]=0;
    while (t32>=1000){
        t32 = t32 - 1000;
        if (t32>=0) pC[4]++;
    }
    pC[4]=pC[4]+0x30; //convert to ascii by adding 0x30

    // do 100
    pC[3]=0;
    while (t32>=100){
        t32 = t32 - 100;
        if (t32>=0) pC[3]++;
    }
    pC[3]=pC[3]+0x30; //convert to ascii by adding 0x30

    // do 10
    pC[2]=0;
    while (t32>=10){
        t32 = t32 - 10;
        if (t32>=0) pC[2]++;
    }
    pC[2]=pC[2]+0x30; //convert to ascii by adding 0x30
    // do 1
    pC[1] = t32 + 0x30;   //remainer + convert to ascii by adding 0x30
}


// ---------------------- ASCII characters for LCD display ----------------------------
// char to two digit HEX display
// Output is in char array pC[]
void char2_HEX(int8_t ch){
    int8_t lwN;
    int8_t upN;
    //split into 2 nibbles
    lwN=ch&0x0F;
    upN=(ch&0xF0)>>4;
    // lower nibble
    if (lwN<=9) {
        pC[1] = lwN + 0x30;   //convert to ascii by adding 0x30
    } else {
        pC[1] = (lwN-10) + 0x41;   //convert to ascii by by starting at "A"
    }
    // upper nibble
    if (upN<=9) {
        pC[2] = upN + 0x30;   //convert to ascii by adding 0x30
    } else {
        pC[2] = (upN-10) + 0x41;   //convert to ascii by by starting at "A"
    }
}




//****************** delay_ms ******************
void delay_ms(unsigned int xdms)
{
    unsigned int xi;
    for (xi = xdms; xi > 0; xi--) {
        __delay_cycles(cM*1000);   //1ms delay
    }
}



//****************** Initialize LCD ******************
void init_LCD(void)
{
    //reset LCD display
    // data sheet says keep low for >=5ms
    for (x = 50; x > 0; x--) {
        __delay_cycles(cM*1000);   //1ms delay
    }
    // reset the LCD display using P3.0
    P2OUT &= ~BIT2;    //force low to reset display
    // data sheet says keep low for >=10ms
    for (x = 10; x > 0; x--) {
        __delay_cycles(cM*1000);   //1ms delay
    }
    P2OUT |= BIT2;      //return to high state
    __delay_cycles(cM*1000);     //delay 1ms (data sheets says >=1ms)
    // display visions SA0=VDD then address 0x7A(8bit) or 0x3D(7bit)
    //  Co 1=continue,   D/C 1=data(ram) 0=command
    i2cm_start();
    i2cm_out(0x7A);  //send address with write byte
    i2cm_out(0x80);  i2cm_out(0x3A);  //control byte ,Function Set: 8 bit data length extension Bet RE=1; REV=0
    i2cm_out(0x80);  i2cm_out(0x09);  //control byte, Extended function set: 4 line Display
    i2cm_out(0x80);  i2cm_out(0x06);  //control byte, Entry Mode Set: bottom view
    i2cm_out(0x80);  i2cm_out(0x1e);  //control byte, Bias Setting: BS1=1
    i2cm_out(0x80);  i2cm_out(0x39);  //control byte, Function Set: 8 bit data length extension Bit RE=0, IS=1
    i2cm_out(0x80);  i2cm_out(0x1b);  //control byte, Internal OSC: BS0=1 -> Bias 1/6
    i2cm_out(0x80);  i2cm_out(0x6e);  //control byte, Follower Control: Divider on and set value
    i2cm_out(0x80);  i2cm_out(0x57);  //control byte, Power Control: Booster on and set contrast (DB1=C5, DB0=C4)
    i2cm_out(0x80);  i2cm_out(0x72);  //control byte, Contrast Set: Set Contrast (DB3-DB0=C3-C0)
    i2cm_out(0x80);  i2cm_out(0x38);  //control byte, Function Set: 8 bet data length extension Bit RE=0; IS=0
    i2cm_out(0x80);  i2cm_out(0x0F);  //control byte, Display on: cursor on, blink on
    i2cm_out(0x00);  i2cm_out(0x01);  //control byte  (no more continuation), Display on: cursor on, blink on
    i2cm_stop();
}




//***********************************************************************************************
//Send data byte out on bang i2c, return false if ack
//Assumes start has been set up or a next byte
//so  both lines are assumed low
// **Lower byte of data is sent**
unsigned char i2cm_out(unsigned int sdata )
{
    //runs at 2.5us between clocks which is 400KHz (likely maximum rate)

    //volatile unsigned int i= 0; //will be register
     //output eight bits of 'data'
 //   for(i=0; i < 8; i++ )
 //   {
        //send the data bit
        if( sdata & 0x80 ) {I2C_MASTER_DIR &= ~SDApin;}  //set for data input (which sets the data high: pull up)
        else  {            I2C_MASTER_DIR |= SDApin;}    //ensure data output direction set
        __delay_cycles(4);              //delay (time for data to settle before clk edge rises)
        I2C_MASTER_DIR &= ~SCLpin;      //SCL high  by changing to input (pullup)
        __delay_cycles(cM/2);           //us of delay: rise time longer than fall time
        sdata= sdata << 1;              //shift next data bit (do shifting here to add delay)
        I2C_MASTER_DIR |= SCLpin;       //SCL as output (makes clk low)

        //send the data bit
        if( sdata & 0x80 ) {I2C_MASTER_DIR &= ~SDApin;}  //set for data input (which sets the data high: pull up)
        else  {            I2C_MASTER_DIR |= SDApin;}    //ensure data output direction set
        __delay_cycles(4);              //delay (time for data to settle before clk edge rises)
        I2C_MASTER_DIR &= ~SCLpin;      //SCL high  by changing to input (pullup)
        __delay_cycles(cM/2);           //us of delay: rise time longer than fall time
        sdata= sdata << 1;              //shift next data bit (do shifting here to add delay)
        I2C_MASTER_DIR |= SCLpin;       //SCL as output (makes clk low)

        //send the data bit
        if( sdata & 0x80 ) {I2C_MASTER_DIR &= ~SDApin;}  //set for data input (which sets the data high: pull up)
        else  {            I2C_MASTER_DIR |= SDApin;}    //ensure data output direction set
        __delay_cycles(4);              //delay (time for data to settle before clk edge rises)
        I2C_MASTER_DIR &= ~SCLpin;      //SCL high  by changing to input (pullup)
        __delay_cycles(cM/2);           //us of delay: rise time longer than fall time
        sdata= sdata << 1;              //shift next data bit (do shifting here to add delay)
        I2C_MASTER_DIR |= SCLpin;       //SCL as output (makes clk low)

        //send the data bit
        if( sdata & 0x80 ) {I2C_MASTER_DIR &= ~SDApin;}  //set for data input (which sets the data high: pull up)
        else  {            I2C_MASTER_DIR |= SDApin;}    //ensure data output direction set
        __delay_cycles(4);              //delay (time for data to settle before clk edge rises)
        I2C_MASTER_DIR &= ~SCLpin;      //SCL high  by changing to input (pullup)
        __delay_cycles(cM/2);           //us of delay: rise time longer than fall time
        sdata= sdata << 1;              //shift next data bit (do shifting here to add delay)
        I2C_MASTER_DIR |= SCLpin;       //SCL as output (makes clk low)

        //send the data bit
        if( sdata & 0x80 ) {I2C_MASTER_DIR &= ~SDApin;}  //set for data input (which sets the data high: pull up)
        else  {            I2C_MASTER_DIR |= SDApin;}    //ensure data output direction set
        __delay_cycles(4);              //delay (time for data to settle before clk edge rises)
        I2C_MASTER_DIR &= ~SCLpin;      //SCL high  by changing to input (pullup)
        __delay_cycles(cM/2);           //us of delay: rise time longer than fall time
        sdata= sdata << 1;              //shift next data bit (do shifting here to add delay)
        I2C_MASTER_DIR |= SCLpin;       //SCL as output (makes clk low)

        //send the data bit
        if( sdata & 0x80 ) {I2C_MASTER_DIR &= ~SDApin;}  //set for data input (which sets the data high: pull up)
        else  {            I2C_MASTER_DIR |= SDApin;}    //ensure data output direction set
        __delay_cycles(4);              //delay (time for data to settle before clk edge rises)
        I2C_MASTER_DIR &= ~SCLpin;      //SCL high  by changing to input (pullup)
        __delay_cycles(cM/2);           //us of delay: rise time longer than fall time
        sdata= sdata << 1;              //shift next data bit (do shifting here to add delay)
        I2C_MASTER_DIR |= SCLpin;       //SCL as output (makes clk low)

        //send the data bit
        if( sdata & 0x80 ) {I2C_MASTER_DIR &= ~SDApin;}  //set for data input (which sets the data high: pull up)
        else  {            I2C_MASTER_DIR |= SDApin;}    //ensure data output direction set
        __delay_cycles(4);              //delay (time for data to settle before clk edge rises)
        I2C_MASTER_DIR &= ~SCLpin;      //SCL high  by changing to input (pullup)
        __delay_cycles(cM/2);           //us of delay: rise time longer than fall time
        sdata= sdata << 1;              //shift next data bit (do shifting here to add delay)
        I2C_MASTER_DIR |= SCLpin;       //SCL as output (makes clk low)

        //send the data bit
        if( sdata & 0x80 ) {I2C_MASTER_DIR &= ~SDApin;}  //set for data input (which sets the data high: pull up)
        else  {            I2C_MASTER_DIR |= SDApin;}    //ensure data output direction set
        __delay_cycles(4);              //delay (time for data to settle before clk edge rises)
        I2C_MASTER_DIR &= ~SCLpin;      //SCL high  by changing to input (pullup)
        __delay_cycles(cM/2);           //us of delay: rise time longer than fall time
        sdata= sdata << 1;              //shift next data bit (do shifting here to add delay)
        I2C_MASTER_DIR |= SCLpin;       //SCL as output (makes clk low)
  //  }

    //check ack
    I2C_MASTER_DIR &= ~SDApin;      //set for data input
    __delay_cycles(cM/4);
    I2C_MASTER_DIR &= ~SCLpin;      //SCL high  by changing to input (pullup)
    __delay_cycles(cM/2);
    //get the ack bit and leave sda in last state
    unsigned char ack= I2C_MASTER_IN & SDApin;
    if( ack ){I2C_MASTER_DIR &= ~SDApin;}      //leave data high (by leaving as data input)
    else     {I2C_MASTER_DIR |= SDApin; }       //data output

    __delay_cycles(cM/2);
    I2C_MASTER_DIR |= SCLpin;       //SCL as output
    __delay_cycles(cM/4);         //no delay required as subroutine offers enough delay
    return ack;
}


// start condition (data changes LOW while clock is high)
void i2cm_start(void)
{
    __delay_cycles(cM);
    I2C_MASTER_DIR &= ~SDApin;      //ensure data is high by ensuring it is set to input

    __delay_cycles(cM);
    I2C_MASTER_DIR &= ~SCLpin;      //SCL high  by changing to input (pullup)

    __delay_cycles(cM);
    I2C_MASTER_OUT &= ~SDApin;      //SDA low (while clk is high)
    I2C_MASTER_DIR |= SDApin;       //data output

    __delay_cycles(cM);
    I2C_MASTER_OUT &= ~SCLpin;      //Set Clock Low
    I2C_MASTER_DIR |= SCLpin;       //SCL as output
    __delay_cycles(cM);
}

//Assumes the clock is low (data changes HIGH while clock is high)
void i2cm_stop(void)
{
    __delay_cycles(cM);
    I2C_MASTER_OUT &= ~SDApin;      //SDA low
    I2C_MASTER_DIR |= SDApin;       //ensure data as output

    __delay_cycles(cM);       //
    I2C_MASTER_DIR &= ~SCLpin;      //SCL high  by changing to input (pullup)

    __delay_cycles(cM);       //
    I2C_MASTER_DIR &= ~SDApin;  //SDA high by changing to input (pullup) (causes STOP condition)

    __delay_cycles(cM);
}














